/*! \file tlk_odrm_api.h
**********************************************************************************
*Title:                        Discretix OMA DRM v2 Toolkit header file
*
* Filename:                     tlk_odrm_api.h
*
* Project, Target, subsystem:   Toolkit, OMA DRM
* 
* Created:                      20.11.2007
*
* Modified:                     20.11.2007
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_ODRM_API_H_
#define _TLK_ODRM_API_H_

#ifdef __cplusplus
extern "C" {
#endif

    /*----------- Doxygen Main Page text -----------------------------------------*/
    /*! \mainpage OMA DRM Toolkit API

    <h1>OMA DRM Toolkit API Documentation</h1>
    <p>
    <img width="268" height="148" src="DxLogo.jpg">
    </p>
    <P class="ConfidentialNotice" style="TEXT-ALIGN: justify">
    This document may be used 
    in its complete form only and is solely for the use of Discretix employees and 
    authorized Discretix channels or customers. The material herein is proprietary 
    to Discretix Ltd., any unauthorized reproduction of any part thereof is 
    strictly prohibited. The contents of this document are believed to be accurate 
    at the time of distribution. Discretix reserves the right to alter this 
    information at any time without notice. The primary distribution media of this 
    document is soft copy.
    </P>

    **/
/*----------- Global Includes ------------------------------------------------*/
#include "CRYS_AES.h"
#include "CRYS_RSA_Types.h"
#include "DX_VOS_BaseTypes.h"

/*----------- Local Includes ------------------------------------------------*/
#include "tlk_odrm_errors.h"
#include "tlk_odrm_types.h"
   
/*----------- Global definitions ---------------------------------------------*/
#define TLK_ODRM_NO_GENERATION_DOMAIN_KEY_INDEX         (0xFFFFFFFF)

#ifdef TLK_ODRM_SEP_SIMULATOR
#define TLK_ODRM_RSA_WRAPPED_KEY_SIZE_IN_BYTES      (2240UL)
#else
#define TLK_ODRM_RSA_WRAPPED_KEY_SIZE_IN_BYTES      (3032UL)
#endif

#define TLK_ODRM_KEY_RESTRICTION  (KMNG_KEY_RESTRICTION_NON_EXPORTABLE)
/*----------- Global function prototypes -------------------------------------*/



/*------------------------------ NEW ODRM API --------------------------------*/
#include "tlk_cert_types.h"
#include "tlk_sclk_api_types.h"
#include "KMNG_Defs.h"

/*TLK_ODRM_ProcessJoinDomain*/
/**
* @brief     This function is invoked when the DRM agent processes a join domain message. 
*			 It lets the license handler securely extract the domain key from the received <domainKey> element.


* @param domainXml					[in]     - The protected domain key XML structure - Note: this should be the canonized version of the XML (in this case the xmlns:roap="urn:oma:bac:dldrm:roap-1.0" attribute should be inserted as the first atribute of the XML roap:domainKey element) 
* @param deviceKeyRing_ptr          [in]     - A pointer to a key ring holding the device private key.
* @param devicePrivateKey_ptr		[in]     - A ponter to the device private key in the key ring above.
* @param keyRing_ptr				[out]    - A pointer to a key ring into which the domain key will be inserted.
* @param domainKeyId_ptr            [out]    - A ponter to the domain key in the key ring above.
* @param workspace_ptr				[in]     - A buffer for internal encryption calculations.
* @param workspaceSizeInBytes       [in]     - Workspace pointer size in bytes
*
* Note: more specific error codes will be added later.
* @return TLK_ODRM_RC_OK                     - The operation Kmac and Kd derivation completed successfully.
* @return TLK_ODRM_RC_FAIL                   - The operation Kmac and Kd derivation failed.
**/
DxError_t TLK_ODRM_ProcessJoinDomain(TLK_ODRM_Buffer_t            *domainXml,   
									 TLK_ODRM_KMNGKeyRing_t      *deviceKeyRing_ptr,
									   TLK_ODRM_KMNGKey_t          *devicePrivateKey_ptr,
									   TLK_ODRM_KMNGKeyRing_t      *keyRing_ptr,
									   TLK_ODRM_KMNGKey_t          *domainKeyId_ptr,
									   DxByte_t                    *workspace_ptr,
									   DxUint32_t                    workspaceSizeInBytes);   



/*TLK_ODRM_VerifyDeviceRoBeforeInstall*/
/**
* @brief     This function is invoked when a license arrives at the device (in the rights-object acquisition protocol). 
*			 It lets the license handler pre-process the protected RO in order to increase performance of the license consumption process.
*			 The function will verify the license and extract the kmac and krec keys into the given key ring using the device key.


* @param roXml_ptr					[in]     - The protected domain key XML structure. Note: this should be the canonized version of the XML (in this case the xmlns:roap="urn:oma:bac:dldrm:roap-1.0" attribute should be inserted as the first attribute of the XML roap:ro element) 
* @param deviceKeyRing_ptr          [in]     - A pointer to a key ring holding the device private key.
* @param devicePrivateKey_ptr		[in]     - A ponter to the device private key in the key ring above.
* @param outputKeyRing_ptr			[out]    - A pointer to a key ring into which the kmac and krec keys key will be inserted.
* @param kMacId_ptr					[out]    - A ponter to the kmac key in the key ring above.
* @param kRecId_ptr					[out]    - A ponter to the krec key in the key ring above.
* @param workspace_ptr				[in]     - A buffer for internal encryption calculations.
* @param workspaceSizeInBytes       [in]     - Workspace pointer size in bytes
*
* Note: more specific error codes will be added later.
* @return TLK_ODRM_RC_OK                     - The operation Kmac and Kd derivation completed successfully.
* @return TLK_ODRM_RC_FAIL                   - The operation Kmac and Kd derivation failed.
**/
DxError_t TLK_ODRM_VerifyDeviceRoBeforeInstall(TLK_ODRM_Buffer_t              *roXml_ptr, 
											TLK_ODRM_KMNGKeyRing_t      *deviceKeyRing_ptr,
											TLK_ODRM_KMNGKey_t          *devicePrivateKey_ptr,
											TLK_ODRM_KMNGKeyRing_t      *outputKeyRing_ptr,
											TLK_ODRM_KMNGKey_t          *kMacId_ptr,
											TLK_ODRM_KMNGKey_t          *kRecId_ptr,
											DxByte_t                    *workspace_ptr,
											DxUint32_t                    workspaceSizeInBytes
											);

/*TLK_ODRM_VerifyDomainRoBeforeInstall*/
/**
* @brief     This function is invoked when a domain license arrives at the device (in the rights-object acquisition protocol). 
*			 It lets the license handler pre-process the protected RO in order to increase performance of the license consumption process.
*			 The function will verify the domain license and extract the kmac and krec keys into the given key ring using the domain key.


* @param roXml_ptr					[in]     - The protected domain key XML structure. Note: this should be the canonized version of the XML (in this case the xmlns:roap="urn:oma:bac:dldrm:roap-1.0" attribute should be inserted as the first attribute of the XML roap:ro element)  
* @param domainKeyRing_ptr          [in]     - A pointer to a key ring holding the domain key.
* @param domainKeyId_ptr			[in]     - A ponter to the domain key in the key ring above.
* @param domainGeneration			[in]	 - A domain generation (relevant for domain resolution).
* @param outputKeyRing_ptr			[out]    - A pointer to a key ring into which the kmac and krec keys key will be inserted.
* @param kMacId_ptr					[out]    - A ponter to the kmac key in the key ring above.
* @param kRecId_ptr					[out]    - A ponter to the krec key in the key ring above.
* @param workspace_ptr				[in]     - A buffer for internal encryption calculations.
* @param workspaceSizeInBytes       [in]     - Workspace pointer size in bytes
*
* Note: more specific error codes will be added later.
* @return TLK_ODRM_RC_OK                     - The operation Kmac and Kd derivation completed successfully.
* @return TLK_ODRM_RC_FAIL                   - The operation Kmac and Kd derivation failed.
**/
DxError_t TLK_ODRM_VerifyDomainRoBeforeInstall(TLK_ODRM_Buffer_t              *roXml_ptr, 
											TLK_ODRM_KMNGKeyRing_t      *domainKeyRing_ptr,
											TLK_ODRM_KMNGKey_t          *domainKeyId_ptr,
											TLK_ODRM_DomainGeneration_t  domainGeneration,
											TLK_ODRM_KMNGKeyRing_t      *outputKeyRing_ptr,
											TLK_ODRM_KMNGKey_t          *kMacId_ptr,
											TLK_ODRM_KMNGKey_t          *kRekId_ptr,
											DxByte_t                    *workspace_ptr,
											DxUint32_t                    workspaceSizeInBytes
											);



/*TLK_ODRM_RetrieveKey*/
/**
* @brief     This function extracts the content encryption key for a specific asset from a license. 
*			 It is invoked when the content needs to be consumed

* @param roXml_ptr					[in]     - The RO XML structure. Note: this should be the canonized version of the XML  (in this case the xmlns:roap="urn:oma:bac:dldrm:roap-1.0" attribute should be inserted as the first attribute of the XML roap:ro element) 
* @param kMaxKRekKeyRing_ptr        [in]     - A pointer to a key ring holding the kmac and krec keys for the asset.
* @param kMacId_ptr					[in]     - A ponter to the kmac key in the key ring above.
* @param kRekId_ptr					[in]     - A ponter to the krec key in the key ring above.
* @param assetId_ptr				[in]     - The ID of the asset for which keys are requested.
* @param serviceDescr_ptr			[in]     - The service clock descriptor indicating the correct secure time to use for the license validation.
* @param roXml_ptr					[in]     - [optional] The parent RO XML structure.
* @param parentKMaxKRekKeyRing_ptr	[in]     - [optional] A pointer to a key ring holding the kmac for the assets parent. 
* @param parentKMacId_ptr			[in]     - [optional] A ponter to the kmac key in the key ring above. 
* @param groupKey					[in]	 - [optional] An optional group key buffer for group protected assets.
* @param contentAesWrappedKey_ptr	[out]	 - The key for the requested asset wrapped.
* @param workspace_ptr				[in]     - A buffer for internal encryption calculations.
* @param workspaceSizeInBytes       [in]     - Workspace pointer size in bytes
*
* Note: more specific error codes will be added later.
* @return TLK_ODRM_RC_OK                     - The operation Kmac and Kd derivation completed successfully.
* @return TLK_ODRM_RC_FAIL                   - The operation Kmac and Kd derivation failed.
**/
DxError_t TLK_ODRM_RetrieveKey(TLK_ODRM_Buffer_t                *roXml_ptr,
							TLK_ODRM_KMNGKeyRing_t              *kMaxKRekKeyRing_ptr,
							TLK_ODRM_KMNGKey_t                  *kMacId_ptr,
							TLK_ODRM_KMNGKey_t                  *kRekId_ptr,
							const DxChar_t                      *assetUid_ptr,
							TLK_SCLK_ServiceClockDescriptor_t   *serviceDescr_ptr,
							TLK_ODRM_Buffer_t                   *parentRoXml_ptr,
							TLK_ODRM_KMNGKeyRing_t              *parentKMaxKRekKeyRing_ptr,
							TLK_ODRM_KMNGKey_t                  *parentKMacId_ptr,
							TLK_ODRM_Buffer_t                   *groupKey,
							TLK_ODRM_GROUP_KEY_ENCRYPTION_TYPE    groupKeyEncryptionType,
							KMNG_AES_WrappedKey_t	             contentAesWrappedKey_ptr,
							DxByte_t                    *workspace_ptr,
							DxUint32_t                    workspaceSizeInBytes
							);


/*TLK_ODRM_WorkspaceSizeGet*/
/*!
\brief Get the minimum workspace size needed per Toolkit OMA DRM operation

@param workspaceOp                  [in]     - Describes the type of Toolkit OMA DRM operation 
@param workspaceMinSize_ptr         [out]    - Minimum workspace size needed

@return TLK_ODRM_RC_OK                       - The operation completed successfully.
@return TLK_ODRM_RC_FAIL                     - The operation failed.
**/
DxError_t TLK_ODRM_WorkspaceSizeGet(                  TLK_ODRMWorkspaceOp_t       workspaceOp,
                                                      DxUint32_t                 *workspaceMinSize_ptr);

#ifdef __cplusplus
}
#endif

#endif /*_TLK_ODRM_API_H*/
